import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Calendar;
import java.util.GregorianCalendar;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.lang.StringUtils;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.X509Extensions;
import org.bouncycastle.asn1.x509.X509Name;
import org.bouncycastle.openssl.PEMWriter;
import org.bouncycastle.x509.X509V3CertificateGenerator;
import org.bouncycastle.x509.extension.SubjectKeyIdentifierStructure;

/**
 * @author michail
 * @version $Rev: 19 $
 */
public final class CertGenerator {

    private static final int YEARS_VALID = 20;

    private final int keySize = 1024;

    @SuppressWarnings("static-access")
    public static void main(final String[] args) throws Exception {
        final Options options = new Options();
        options.addOption(OptionBuilder.withArgName("?").withDescription(
                                                                         "print this help message").create(
                                                                                                           "help"));
        options.addOption(OptionBuilder.withArgName("keyfile").isRequired().hasArg().withDescription(
                                                                                                     "the name of the private key file").create(
                                                                                                                                                "keyfile"));
        options.addOption(OptionBuilder.withArgName("certfile").isRequired().hasArg().withDescription(
                                                                                                      "the name of the certification file").create(
                                                                                                                                                   "certfile"));

        options.addOption(OptionBuilder.withArgName("keystore").isRequired().hasArg().withDescription(
                                                                                                      "the name of the keystore file").create(
                                                                                                                                              "keystore"));
        options.addOption(OptionBuilder.withArgName("storepass").isRequired().hasArg().withDescription(
                                                                                                       "the keystore password").create(
                                                                                                                                       "storepass"));
        options.addOption(OptionBuilder.withArgName("hostname").isRequired().hasArg().withDescription(
                                                                                                      "the hostname ").create(
                                                                                                                              "hostname"));
        options.addOption(OptionBuilder.withArgName("url").hasArg().withDescription(
                                                                                    "the url path of the idp (default: /idp/shibboleth)").create(
                                                                                                                                                 "url"));

        final CommandLineParser parser = new PosixParser();

        try {
            final CommandLine cmd = parser.parse(options, args);

            if (cmd.hasOption("help")
                    || (cmd.getOptions() == null || cmd.getOptions().length == 0)) {
                printHelp(options);
            } else {
                final String hostName = cmd.getOptionValue("hostname");
                final String url = StringUtils.defaultIfEmpty(
                                                              cmd.getOptionValue("url"),
                                                              "/idp/shibboleth");
                final String keyFileName = cmd.getOptionValue("keyfile");
                final String certFileName = cmd.getOptionValue("certfile");
                final String keyStoreFileName = cmd.getOptionValue("keystore");
                final String keyStorePass = cmd.getOptionValue("storepass");
                new CertGenerator(hostName, url, new File(keyFileName),
                        new File(certFileName), new File(keyStoreFileName),
                        keyStorePass);
            }
        } catch (ParseException e) {
            printHelp(options);
        }
    }

    private static void printHelp(final Options options) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp(CertGenerator.class.getName(), options, true);
    }

    private CertGenerator(final String hostname, final String url,
            final File privateKeyFile, final File certificateFile,
            final File keyStoreFile, final String keyStorePass)
            throws NoSuchAlgorithmException, InvalidKeyException,
            IllegalStateException, NoSuchProviderException,
            SignatureException, KeyStoreException, CertificateException,
            IOException {

        //        final Provider provider = new BouncyCastleProvider();
        //        Security.addProvider(provider);

        final KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("RSA");
        keyPairGenerator.initialize(this.keySize);
        final KeyPair keypair = keyPairGenerator.generateKeyPair();
        X509Certificate cert;

        final X509V3CertificateGenerator certGen = new X509V3CertificateGenerator();
        certGen.setPublicKey(keypair.getPublic());

        final X509Name dn = new X509Name(false, "CN=" + hostname);
        certGen.setIssuerDN(dn);
        certGen.setSubjectDN(dn);

        final ASN1EncodableVector list = new ASN1EncodableVector();
        list.add(new GeneralName(GeneralName.dNSName, hostname));
        list.add(new GeneralName(GeneralName.uniformResourceIdentifier,
                "https://" + hostname + url));
        certGen.addExtension(X509Extensions.SubjectAlternativeName, false,
                             new GeneralNames(new DERSequence(list)));
        certGen.addExtension(X509Extensions.SubjectKeyIdentifier, false,
                             new SubjectKeyIdentifierStructure(
                                     keypair.getPublic()));

        final GregorianCalendar date = new GregorianCalendar();
        certGen.setNotBefore(date.getTime());

        date.set(Calendar.YEAR, date.get(Calendar.YEAR) + YEARS_VALID);
        certGen.setNotAfter(date.getTime());

        certGen.setSerialNumber(new BigInteger(256, new SecureRandom()));

        certGen.setSignatureAlgorithm("SHA1withRSA");
        cert = certGen.generate(keypair.getPrivate());

        final String keyAlias = hostname;
        final KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());

        // empty keystore
        keyStore.load(null, keyStorePass.toCharArray());
        final Certificate[] certs = new Certificate[1];
        certs[0] = cert;
        keyStore.setKeyEntry(keyAlias, keypair.getPrivate(),
                             keyStorePass.toCharArray(), certs);

        if (!privateKeyFile.createNewFile()) {
            throw new IOException("could not create new file "
                    + keyStoreFile.getPath());
        }
        PEMWriter keyOut = new PEMWriter(new FileWriter(privateKeyFile));
        keyOut.writeObject(keypair.getPrivate());
        keyOut.flush();
        keyOut.close();

        if (!certificateFile.createNewFile()) {
            throw new IOException("could not create new file "
                    + keyStoreFile.getPath());
        }
        keyOut = new PEMWriter(new FileWriter(certificateFile));
        keyOut.writeObject(cert);
        keyOut.flush();
        keyOut.close();

        if (!keyStoreFile.createNewFile()) {
            throw new IOException("could not create new file "
                    + keyStoreFile.getPath());
        }
        final FileOutputStream fileOutputStream = new FileOutputStream(
                keyStoreFile);
        keyStore.store(fileOutputStream, keyStorePass.toCharArray());
        fileOutputStream.flush();
        fileOutputStream.close();

    }
}
